# Datok - Finite State Tokenizer

[![DOI](https://zenodo.org/badge/416456182.svg)](https://zenodo.org/badge/latestdoi/416456182)

![Introduction to Datok](https://raw.githubusercontent.com/KorAP/Datok/master/misc/introducing-datok.gif)

Implementation of a finite state automaton for
high-performance large-scale natural language tokenization,
based on a finite state
transducer generated with [Foma](https://fomafst.github.io/).

The repository currently contains precompiled tokenizer models for

- [german](testdata/tokenizer_de.matok)
- [english](testdata/tokenizer_en.matok)

The focus of development is on the tokenization of
[DeReKo](https://www.ids-mannheim.de/digspra/kl/projekte/korpora),
the german reference corpus.

Datok can be used as a standalone tool or as a library in Go.

## Performance

![Speed comparison of german tokenizers](https://raw.githubusercontent.com/KorAP/Datok/master/misc/benchmarks.svg)

Chart showing speed comparison of different tokenizers and sentence splitters
for German. `Effi` refers to tokenizing and/or sentence splitting of one
issue of [Effi Briest](https://www.gutenberg.org/cache/epub/5323/pg5323.html).
Datok is optimized for large batch sizes, while other tools may
perform better in other scenarios.
For further benchmarks, especially regarding the quality of tokenization,
see Diewald/Kupietz/Lüngen (2022).

## Tokenization

```
Usage: datok tokenize --tokenizer=STRING <input>

Arguments:
  <input>    Input file to tokenize (use - for STDIN)

Flags:
  -h, --help                  Show context-sensitive help.

  -t, --tokenizer=STRING      The Matrix or Double Array Tokenizer file
      --[no-]tokens           Print token surfaces (defaults to true)
      --[no-]sentences        Print sentence boundaries (defaults to true)
  -p, --token-positions       Print token offsets (defaults to false)
      --sentence-positions    Print sentence offsets (defaults to false)
      --newline-after-eot     Ignore newline after EOT (defaults to false)
```

The special `END OF TRANSMISSION` character (`\x04`) can be used to mark the end of a text.

> *Caution*: When experimenting with STDIN and echo,
> you may need to disable [history expansion](https://www.gnu.org/software/bash/manual/html_node/History-Interaction.html).


## Conversion

```
Usage: datok convert --foma=STRING --tokenizer=STRING

Flags:
  -h, --help                Show context-sensitive help.

  -i, --foma=STRING         The Foma FST file
  -o, --tokenizer=STRING    The Tokenizer file
  -d, --double-array        Convert to Double Array instead of Matrix
                            representation
```

## Library

```go
package main

import (
	"github.com/KorAP/datok"
	"os"
	"strings"
)

func main () {

	// Load transducer binary
	dat := datok.LoadTokenizerFile("tokenizer_de.matok")
	if dat == nil {
		panic("Can't load tokenizer")
	}

	// Create a new TokenWriter object
	tw := datok.NewTokenWriter(os.Stdout, datok.TOKENS|datok.SENTENCES)
	defer tw.Flush()

	// Create an io.Reader object refering to the data to tokenize
	r := strings.NewReader("Das ist <em>interessant</em>!")

	// The transduceTokenWriter accepts an io.Reader
	// object and a TokenWriter object to transduce the input
	dat.TransduceTokenWriter(r, tw)
}
```

## Conventions

The FST generated by [Foma](https://fomafst.github.io/) must adhere to
the following rules, to be convertible by Datok:

- Character accepting arcs need to be translated
  *only* to themselves or to ε (the empty symbol).
  I.e. they will either be unchanged part of the
  output or ignored (e.g. whitespace characters).
- Multi-character symbols are not allowed,
  except for the `@_TOKEN_BOUND_@`,
  that denotes the end of a token.
- ε accepting arcs (transitions not consuming
  any character) need to be translated to
  the `@_TOKEN_BOUND_@`.
- Two consecutive `@_TOKEN_BOUND_@`s mark a sentence end.
- Flag diacritics are not supported.
- Final states are ignored. The `@_TOKEN_BOUND_@` marks
  the end of a token instead.

A minimal usable tokenizer written in XFST and following
the guidelines to tokenizers in Beesley and Karttunen (2003)
and Beesley (2004) would look like this:

```xfst
define TB "@_TOKEN_BOUND_@";

define WS [" "|"\u000a"|"\u0009"];

define PUNCT ["."|"?"|"!"];

define Char \[WS|PUNCT];

define Word Char+;

! Compose token bounds
define Tokenizer [[Word|PUNCT] @-> ... TB] .o.
! Compose Whitespace ignorance
       [WS+ @-> 0] .o.
! Compose sentence ends
       [[PUNCT+] @-> ... TB \/ TB _ ];

read regex Tokenizer;
```

> *Hint*: For development in Foma it's easier to replace
> `@_TOKEN_BOUND_@` with a newline symbol.

## Building

To build the tokenizer tool, run

```shell
$ make build
```

To create a foma file from the example sources, first install
[Foma](https://fomafst.github.io/), then run in
the root directory of this repository

```shell
$ cd src && \
  foma -e "source de/tokenizer.xfst" \
  -e "save stack ../mytokenizer.fst" -q -s && \
  cd ..
```

This will load and compile the german `tokenizer.xfst`
and will save the compiled FST as `mytokenizer.fst`
in the root directory.

To generate a Datok FSA (matrix representation) based on
this FST, run

```shell
$ datok convert -i mytokenizer.fst -o mytokenizer.datok
```

To generate a Datok FSA (double array representation*) based
on this FST, run

```shell
$ datok convert -i mytokenizer.fst -o mytokenizer.datok -d
```

The final datok file can then be used as a model for the tokenizer.

* This may take quite some time depending on the number
of arcs in the FST and is therefore not recommended in most cases.


## Technology

Internally the FSA is represented
either as a matrix or as a double array.

Both representations mark all non-word-character targets with a
leading bit. All ε (aka *tokenend*) transitions mark the end of a
token or the end of a sentence (2 subsequential ε).
The transduction is greedy with a single backtracking
option to the last ε transition.

The double array representation (Aoe 1989) of all transitions
in the FST is implemented as an extended DFA following Mizobuchi
et al. (2000) and implementation details following Kanda et al. (2018).


## References

Please cite this work as:

> Diewald, Nils (2022): [*Matrix and Double-Array Representations
> for Efficient Finite State Tokenization*](http://www.lrec-conf.org/proceedings/lrec2022/workshops/CMLC10/pdf/2022.cmlc10-1.4.pdf).
> In: Proceedings of the
> 10th Workshop on Challenges in the Management of Large Corpora
> (CMLC-10) at LREC 2022. Marseille, France, pp. 20-26.

The library contains sources for a german tokenizer
based on [KorAP-Tokenizer](https://github.com/KorAP/KorAP-Tokenizer).

For speed and quality analysis in comparison to other tokenizers for German,
please refer to this article:

> Diewald, Nils, Marc Kupietz, Harald Lüngen (2022): *Tokenizing on scale -
> Preprocessing large text corpora on the lexical and sentence level*.
> In: Proceedings of EURALEX 2022. Mannheim, Germany, pp. 208-221.

The benchmarks can be reproduced using
[this test suite](https://github.com/KorAP/Tokenizer-Evaluation).


## License

Datok is published under the [Apache 2.0 License](LICENSE).

The german and english tokenizer shipped is based on work done by the
[Lucene project](https://github.com/apache/lucene-solr)
(published under the Apache License),
[David Hall](https://github.com/dlwh/epic)
(published under the Apache License),
[Çağrı Çöltekin](https://github.com/coltekin/TRmorph/)
(published under the MIT License),
and [Marc Kupietz](https://github.com/KorAP/KorAP-Tokenizer)
(published under the Apache License).

The english clitics list is based on Zwicky & Pullum (1983).

The foma parser is based on
[*foma2js*](https://github.com/mhulden/foma),
written by Mans Hulden (published under the Apache License).


## Bibliography

Aoe, Jun-ichi (1989): *An Efficient Digital Search Algorithm by Using a Double-Array Structure*.
IEEE Transactions on Software Engineering, 15 (9), pp. 1066-1077.

Beesley, Kenneth R. & Lauri Karttunen (2003): *Finite State Morphology*. Stanford, CA: CSLI Publications.

Beesley, Kenneth R. (2004): *Tokenizing Transducers*.
[https://web.stanford.edu/~laurik/fsmbook/clarifications/tokfst.html](https://web.stanford.edu/~laurik/fsmbook/clarifications/tokfst.html)

Hulden, Mans (2009): *Foma: a finite-state compiler and library*. In: Proceedings of the
12th Conference of the European Chapter of the Association for Computational Linguistics,
Association for Computational Linguistics, pp. 29-32.

Mizobuchi, Shoji, Toru Sumitomo, Masao Fuketa & Jun-ichi Aoe (2000):
*An efficient representation for implementing finite state machines based on the double-array*.
Information Sciences 129, pp. 119-139.

Kanda, Shunsuke, Yuma Fujita, Kazuhiro Morita & Masao Fuketa (2018):
*Practical rearrangement methods for dynamic double-array dictionaries*.
Software: Practice and Experience (SPE), 48(1), pp. 65–83.

Zwicky, Arnold M., Geoffrey K. Pullum (1983):
*Cliticization vs. Inflection: English N’T*.
Language, 59, pp. 502-513.
