/*
 * Decompiled with CFR 0.152.
 */
package de.ids_mannheim.korap.index;

import de.ids_mannheim.korap.KrillIndex;
import de.ids_mannheim.korap.util.KrillProperties;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.MissingOptionException;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.store.MMapDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Indexer {
    private KrillIndex index;
    private int count;
    private int commitCount;
    private boolean progressEnabled = false;
    private SimpleProgressBar progressBar;
    private static String path = null;
    private static boolean addInsteadOfUpsert = false;
    private Pattern jsonFilePattern;
    private Pattern plainJsonFilePattern;
    private static final Logger log = LoggerFactory.getLogger(Indexer.class);
    private static final boolean DEBUG = false;

    public Indexer(Properties prop) throws IOException {
        if (path == null) {
            path = prop.getProperty("krill.indexDir");
        }
        log.info("Output directory: " + path);
        String commitCount = prop.getProperty("krill.index.commit.count", "1000");
        this.index = new KrillIndex(new MMapDirectory(Paths.get(path, new String[0])));
        this.count = 0;
        this.commitCount = Integer.parseInt(commitCount);
        this.jsonFilePattern = Pattern.compile(".*\\.json\\.gz$");
        this.plainJsonFilePattern = Pattern.compile(".*\\.json$");
    }

    private void initProgress(long total) {
        if (total > 0L) {
            this.progressEnabled = true;
            this.progressBar = new SimpleProgressBar(total);
        }
    }

    private void stepProgress() {
        if (this.progressEnabled && this.progressBar != null) {
            this.progressBar.step();
        }
    }

    private void finishProgress() {
        if (this.progressEnabled && this.progressBar != null) {
            this.progressBar.finish();
        }
    }

    private void parse(File dir) {
        for (String string : dir.list()) {
            Matcher matcher = this.jsonFilePattern.matcher(string);
            if (matcher.find()) {
                String string2 = dir.getPath() + "/" + string;
                try {
                    if (addInsteadOfUpsert) {
                        if (!this.progressEnabled) {
                            log.info("{} Add {} to the index. ", (Object)this.count, (Object)string2);
                        }
                        if (this.index.addDoc(new FileInputStream(string2), true) == null) {
                            log.warn("fail.");
                            continue;
                        }
                    } else {
                        if (!this.progressEnabled) {
                            log.info("{} Add or update {} to the index. ", (Object)this.count, (Object)string2);
                        }
                        if (this.index.upsertDoc(new FileInputStream(string2), true) == null) {
                            log.warn("fail.");
                            continue;
                        }
                    }
                    ++this.count;
                    if (this.count % this.commitCount == 0) {
                        this.commit();
                    }
                    this.stepProgress();
                }
                catch (FileNotFoundException e) {
                    log.error("File " + string2 + " is not found!");
                }
                continue;
            }
            log.warn("Skip " + string + " since it does not have json.gz format.");
        }
    }

    private void parseTar(File tarFile) {
        try {
            InputStream fileInputStream = new FileInputStream(tarFile);
            if (tarFile.getName().toLowerCase().endsWith(".tar.gz") || tarFile.getName().toLowerCase().endsWith(".tgz")) {
                fileInputStream = new GzipCompressorInputStream(fileInputStream);
            }
            try (TarArchiveInputStream tarInputStream = new TarArchiveInputStream(fileInputStream);){
                TarArchiveEntry entry;
                while ((entry = tarInputStream.getNextTarEntry()) != null) {
                    if (entry.isDirectory()) continue;
                    String entryName = entry.getName();
                    Matcher gzipMatcher = this.jsonFilePattern.matcher(entryName);
                    Matcher plainMatcher = this.plainJsonFilePattern.matcher(entryName);
                    boolean isGzipped = gzipMatcher.find();
                    boolean isPlainJson = plainMatcher.find();
                    if (isGzipped || isPlainJson) {
                        int bytesRead;
                        byte[] entryData = new byte[(int)entry.getSize()];
                        for (int totalRead = 0; totalRead < entryData.length && (bytesRead = tarInputStream.read(entryData, totalRead, entryData.length - totalRead)) != -1; totalRead += bytesRead) {
                        }
                        ByteArrayInputStream entryStream = new ByteArrayInputStream(entryData);
                        try {
                            if (addInsteadOfUpsert) {
                                if (!this.progressEnabled) {
                                    log.info("{} Add {} from tar {} to the index. ", this.count, entryName, tarFile.getName());
                                }
                                if (this.index.addDoc(entryStream, isGzipped) == null) {
                                    log.warn("fail.");
                                    continue;
                                }
                            } else {
                                if (!this.progressEnabled) {
                                    log.info("{} Add or update {} from tar {} to the index. ", this.count, entryName, tarFile.getName());
                                }
                                if (this.index.upsertDoc(entryStream, isGzipped) == null) {
                                    log.warn("fail.");
                                    continue;
                                }
                            }
                            ++this.count;
                            if (this.count % this.commitCount == 0) {
                                this.commit();
                            }
                            this.stepProgress();
                            continue;
                        }
                        finally {
                            ((InputStream)entryStream).close();
                            continue;
                        }
                    }
                    log.warn("Skip " + entryName + " from tar " + tarFile.getName() + " since it does not have .json or .json.gz format.");
                }
            }
        }
        catch (IOException e) {
            log.error("Error reading tar file " + tarFile.getName(), e);
        }
    }

    private void parseZip(File zipFile) {
        try (ZipFile zip = new ZipFile(zipFile);){
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                if (entry.isDirectory()) continue;
                String entryName = entry.getName();
                Matcher gzipMatcher = this.jsonFilePattern.matcher(entryName);
                Matcher plainMatcher = this.plainJsonFilePattern.matcher(entryName);
                boolean isGzipped = gzipMatcher.find();
                boolean isPlainJson = plainMatcher.find();
                if (isGzipped || isPlainJson) {
                    try (InputStream entryStream = zip.getInputStream(entry);){
                        if (addInsteadOfUpsert) {
                            if (!this.progressEnabled) {
                                log.info("{} Add {} from zip {} to the index. ", this.count, entryName, zipFile.getName());
                            }
                            if (this.index.addDoc(entryStream, isGzipped) == null) {
                                log.warn("fail.");
                                continue;
                            }
                        } else {
                            if (!this.progressEnabled) {
                                log.info("{} Add or update {} from zip {} to the index. ", this.count, entryName, zipFile.getName());
                            }
                            if (this.index.upsertDoc(entryStream, isGzipped) == null) {
                                log.warn("fail.");
                                continue;
                            }
                        }
                        ++this.count;
                        if (this.count % this.commitCount == 0) {
                            this.commit();
                        }
                        this.stepProgress();
                    }
                    catch (IOException e) {
                        log.error("Error reading entry " + entryName + " from zip file " + zipFile.getName(), e);
                    }
                    continue;
                }
                log.warn("Skip " + entryName + " from zip " + zipFile.getName() + " since it does not have .json or .json.gz format.");
            }
        }
        catch (IOException e) {
            log.error("Error reading zip file " + zipFile.getName(), e);
        }
    }

    private void commit() {
        log.info("Committing index ... ");
        try {
            this.index.commit();
        }
        catch (IOException e) {
            log.error("Unable to commit to index " + path);
        }
    }

    private void closeIndex() throws IOException {
        this.commit();
        this.index.closeReader();
        this.index.closeWriter();
    }

    public static void main(String[] argv) {
        Options options = new Options();
        options.addOption(Option.builder("c").longOpt("config").desc("configuration file (defaults to krill.properties).").hasArg().argName("properties file").required().build());
        options.addOption(Option.builder("i").longOpt("input").desc("input paths separated by semicolons. Can be directories containing <filename>.json.gz files, zip files containing .json or .json.gz files, or tar files (including .tar.gz) containing .json or .json.gz files. The indexer will automatically detect the type.").hasArgs().argName("input paths").required().valueSeparator(Character.valueOf(';').charValue()).build());
        options.addOption(Option.builder("o").longOpt("outputDir").desc("index output directory (defaults to krill.indexDir in the configuration.").hasArg().argName("output directory").build());
        options.addOption(Option.builder("a").longOpt("addInsteadofUpsert").desc("Always add files to the index, never update").build());
        options.addOption(Option.builder().longOpt("progress").desc("Show progress bar with ETA").build());
        DefaultParser parser = new DefaultParser();
        String propFile = null;
        String[] inputPaths = null;
        boolean showProgress = false;
        try {
            CommandLine cmd = parser.parse(options, argv);
            log.info("Configuration file: " + cmd.getOptionValue("c"));
            propFile = cmd.getOptionValue("c");
            log.info("Input paths: " + StringUtils.join((Object[])cmd.getOptionValues("i"), ";"));
            inputPaths = cmd.getOptionValues("i");
            if (cmd.hasOption("o")) {
                log.info("Output directory: " + cmd.getOptionValue("o"));
                path = cmd.getOptionValue("o");
            }
            if (cmd.hasOption("a")) {
                addInsteadOfUpsert = true;
            }
            if (cmd.hasOption("progress")) {
                showProgress = true;
            }
        }
        catch (MissingOptionException e) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("Krill indexer\n java -jar -c <properties file> -i <input paths> [-o <output directory> -a --progress]", options);
            return;
        }
        catch (ParseException e) {
            log.error("Unexpected error: " + String.valueOf(e));
            e.printStackTrace();
        }
        Properties prop = KrillProperties.loadProperties(propFile);
        try {
            long total;
            Indexer indexer = new Indexer(prop);
            if (KrillProperties.maxTextSize > 20000000) {
                log.info("Setting max text length to " + KrillProperties.maxTextSize);
                indexer.index.setMaxStringLength(KrillProperties.maxTextSize);
            }
            if (showProgress && (total = Indexer.countTargetFiles(inputPaths)) > 0L) {
                indexer.initProgress(total);
            }
            for (String arg : inputPaths) {
                File f = new File(arg);
                if (f.isDirectory()) {
                    log.info("Indexing files in directory " + arg);
                    indexer.parse(f);
                    continue;
                }
                if (f.isFile() && f.getName().toLowerCase().endsWith(".zip")) {
                    log.info("Indexing files in zip " + arg);
                    indexer.parseZip(f);
                    continue;
                }
                if (f.isFile() && (f.getName().toLowerCase().endsWith(".tar") || f.getName().toLowerCase().endsWith(".tar.gz") || f.getName().toLowerCase().endsWith(".tgz"))) {
                    log.info("Indexing files in tar " + arg);
                    indexer.parseTar(f);
                    continue;
                }
                log.warn("Skipping " + arg + " - not a valid directory, zip file, or tar file");
            }
            indexer.finishProgress();
            indexer.closeIndex();
            log.info("Finished indexing.");
            Object message = "Added ";
            if (!addInsteadOfUpsert) {
                message = (String)message + "or updated ";
            }
            message = (String)message + indexer.count + " file";
            if (indexer.count > 1) {
                message = (String)message + "s";
            }
            System.out.println((String)message + ".");
        }
        catch (IOException e) {
            log.error("Unexpected error: " + String.valueOf(e));
            e.printStackTrace();
        }
    }

    public static long countTargetFiles(String[] inputPaths) {
        if (inputPaths == null) {
            return 0L;
        }
        Pattern gzPattern = Pattern.compile(".*\\.json\\.gz$");
        Pattern jsonPattern = Pattern.compile(".*\\.json$");
        long total = 0L;
        for (String arg : inputPaths) {
            File f = new File(arg);
            if (f.isDirectory()) {
                String[] list = f.list();
                if (list == null) continue;
                for (String name : list) {
                    if (!gzPattern.matcher(name).find()) continue;
                    ++total;
                }
                continue;
            }
            if (f.isFile() && f.getName().toLowerCase().endsWith(".zip")) {
                try (ZipFile zip = new ZipFile(f);){
                    Enumeration<? extends ZipEntry> entries = zip.entries();
                    while (entries.hasMoreElements()) {
                        String entryName;
                        ZipEntry entry = entries.nextElement();
                        if (entry.isDirectory() || !gzPattern.matcher(entryName = entry.getName()).find() && !jsonPattern.matcher(entryName).find()) continue;
                        ++total;
                    }
                }
                catch (IOException e) {
                    log.warn("Unable to count entries in zip " + arg, e);
                }
                continue;
            }
            if (!f.isFile() || !f.getName().toLowerCase().endsWith(".tar") && !f.getName().toLowerCase().endsWith(".tar.gz") && !f.getName().toLowerCase().endsWith(".tgz")) continue;
            try (FileInputStream fis = new FileInputStream(f);
                 GzipCompressorInputStream in = f.getName().toLowerCase().endsWith(".tar.gz") || f.getName().toLowerCase().endsWith(".tgz") ? new GzipCompressorInputStream(fis) : fis;
                 TarArchiveInputStream tis = new TarArchiveInputStream(in);){
                TarArchiveEntry entry;
                while ((entry = tis.getNextTarEntry()) != null) {
                    String entryName;
                    if (entry.isDirectory() || !gzPattern.matcher(entryName = entry.getName()).find() && !jsonPattern.matcher(entryName).find()) continue;
                    ++total;
                }
            }
            catch (IOException e) {
                log.warn("Unable to count entries in tar " + arg, e);
            }
        }
        return total;
    }

    private static class SimpleProgressBar {
        private final long total;
        private long current = 0L;
        private final long startTimeMs;
        private final int barWidth = 40;

        SimpleProgressBar(long total) {
            this.total = total;
            this.startTimeMs = System.currentTimeMillis();
            this.render();
        }

        void step() {
            ++this.current;
            this.render();
        }

        void finish() {
            this.current = Math.max(this.current, this.total);
            this.render();
            System.err.println();
        }

        private void render() {
            double percent = this.total > 0L ? (double)this.current / (double)this.total : 0.0;
            int filled = (int)Math.round(percent * 40.0);
            StringBuilder bar = new StringBuilder(40);
            for (int i = 0; i < 40; ++i) {
                bar.append(i < filled ? (char)'=' : '-');
            }
            long now = System.currentTimeMillis();
            double elapsedSec = (double)(now - this.startTimeMs) / 1000.0;
            double rate = elapsedSec > 0.0 ? (double)this.current / elapsedSec : 0.0;
            long etaSec = rate > 0.0 && this.total > this.current ? (long)Math.ceil((double)(this.total - this.current) / rate) : 0L;
            String etaStr = SimpleProgressBar.formatDuration(etaSec);
            String pctStr = String.format(Locale.US, "%5.1f%%", percent * 100.0);
            String rateStr = String.format(Locale.US, "%.1f/s", rate);
            String line = String.format(Locale.US, "\r[%s] %s %d/%d | %s | ETA %s", bar, pctStr, this.current, this.total, rateStr, etaStr);
            System.err.print(line);
        }

        private static String formatDuration(long seconds) {
            long h = seconds / 3600L;
            long m = seconds % 3600L / 60L;
            long s = seconds % 60L;
            if (h > 99L) {
                return String.format(Locale.US, ">99h", new Object[0]);
            }
            if (h > 0L) {
                return String.format(Locale.US, "%02d:%02d:%02d", h, m, s);
            }
            return String.format(Locale.US, "%02d:%02d", m, s);
        }
    }
}

