package de.ids_mannheim.korapxmltools

import org.junit.After
import org.junit.Before
import java.io.ByteArrayOutputStream
import java.io.PrintStream
import java.net.URL
import kotlin.test.Test
import kotlin.test.assertContains
import de.ids_mannheim.korapxmltools.KorapXml2Conllu

class KorapXml2ConlluTest {
    private val outContent = ByteArrayOutputStream(10000000)
    private val errContent = ByteArrayOutputStream()
    private val originalOut: PrintStream = System.out
    private val originalErr: PrintStream = System.err

    val goe = loadResource("goe.zip").path
    val goeMarmot = loadResource("goe.marmot.zip").path
    val goeTreeTagger = loadResource("goe.tree_tagger.zip").path

    @Before
    fun setUpStreams() {
        System.setOut(PrintStream(outContent))
        System.setErr(PrintStream(errContent))
    }

    @After
    fun restoreStreams() {
        System.setOut(originalOut)
        System.setErr(originalErr)
    }

    private fun loadResource(path: String): URL {
        val resource = Thread.currentThread().contextClassLoader.getResource(path)
        requireNotNull(resource) { "Resource $path not found" }
        return resource
    }

    @Test
    fun canConvertGOE() {
        val classUnderTest = KorapXml2Conllu()
        val args = arrayOf(loadResource("goe.zip").path)
        classUnderTest.main(args)
        assertContains(
            outContent.toString(),
            "# start_offsets = 55 55 59 63 70 75 82 87 94 102 105 111 120 124 130 134 140 144 151 153 163 175 187 191 207 209 213 218 222 239 248 255 259 264 267 271 277 283 297 307"
        )
    }
    @Test
    fun canConvertWithMorphoAnnotations() {
        val classUnderTest = KorapXml2Conllu()
        val args = arrayOf(loadResource("goe.zip").path, loadResource("goe.tree_tagger.zip").path)
        classUnderTest.main(args)
        assertContains(
            outContent.toString(),
            "9\tentzücke\tentzücken\t_\tVVFIN\t_\t_\t_\t_\t1.000000"
        )
    }
    @Test
    fun canInferBaseName() {
        val classUnderTest = KorapXml2Conllu()
        val args = arrayOf(goeTreeTagger)
        classUnderTest.main(args)
        assertContains(
            outContent.toString(),
            "9\tentzücke\tentzücken\t_\tVVFIN\t_\t_\t_\t_\t1.000000"
        )
    }

    @Test
    fun canConvertWfdWithMorphoAnnotations() {
        val classUnderTest = KorapXml2Conllu()
        val args = arrayOf(loadResource("wdf19.zip").path, loadResource("wdf19.tree_tagger.zip").path)
        System.setOut(PrintStream(outContent))
        classUnderTest.main(args)
        assertContains(
            outContent.toString(),
            "30\tvraie\tvrai\t_\tADJ\t_\t_\t_\t_\t1.000000"
        )
    }

    @Suppress("for some reason not working")
    fun canConvertMorphoFeatureAnnotations() {
        val classUnderTest = KorapXml2Conllu()
        val args = arrayOf(goe, goeMarmot)
        classUnderTest.main(args)
        assertContains(
            outContent.toString(),
            "9\tentzücke\tentzücken\t_\tVVFIN\tnumber=sg|person=3|tense=pres|mood=subj\t_\t_\t_\t1.000000"
        )
    }
}
